/** @file   mp3file.cpp
 * @brief   Implementation of MP3File class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:40 $
 * @author  Tomi Lamminsaari
 */

#include "mp3file.h"
#include <almp3.h>

namespace eng2d {


//********************************************************************
//                                                                   *
//      Constructors, destructor and operators                       *
//                                                                   *
//********************************************************************

/** Constructor
 */
MP3File::MP3File( const std::string& file ) :
  m_filename( file ),
  m_filesize( 0 ),
  m_pMP3( 0 ),
  m_pData( 0 ),
  m_open( false )
{
}



/** Destructor
 */
MP3File::~MP3File()
{
  this->close();
}



//********************************************************************
//                                                                   *
//      Public interface                                             *
//                                                                   *
//********************************************************************

/** Loads the given mp3-file.
  */
long MP3File::open()
{
  if ( m_open ) {
    return m_filesize;
  }
  
  m_filesize = file_size( m_filename.c_str() );
  if ( m_filesize == 0 ) {
    // File not found or didn't open
    return 0;
  }
  
  // Allocate the buffer for the MP3-file and load it.
  m_pData = new char[m_filesize];
  if ( m_pData == 0 ) {
    return 0;
  }
  
  PACKFILE* f = pack_fopen( m_filename.c_str(), "r" );
  if ( f == 0 ) {
    this->close();
    return 0;
  }
  pack_fread( m_pData, m_filesize, f );
  pack_fclose(f);
  
  // And create the mp3-structure for the player
  m_pMP3 = almp3_create_mp3( m_pData, m_filesize );
  if ( m_pMP3 == 0 ) {
    this->close();
    return 0;
  }
  
  m_open = true;
  
  // Opening was OK.
  return m_filesize;
}



/** Closes the MP3-file
 */
void MP3File::close()
{
  if ( m_pMP3 != 0 ) {
    almp3_destroy_mp3( m_pMP3 );
    m_pMP3 = 0;
  }
  if ( m_pData != 0 ) {
    delete [] m_pData;
    m_pData = 0;
  }
  m_filesize = 0;
}



//********************************************************************
//                                                                   *
//      Public GET - methods                                         *
//                                                                   *
//********************************************************************

/** Returns pointer to MP3-structure needed by the player
 */
ALMP3_MP3* MP3File::getMP3() const
{
  return m_pMP3;
}



//********************************************************************
//                                                                   *
//      Private methods                                              *
//                                                                   *
//********************************************************************

}
